/**
 * LayerPay Checkout JavaScript
 * Handles wallet connection, network switching, and crypto payments
 * Web3Modal is loaded as ES module in the page head
 */

(function() {
    'use strict';

    // Prevent multiple script loads
    if (window.LayerPay && window.LayerPay._initialized) {
        console.log('LayerPay already initialized, skipping');
        return;
    }

    const LayerPay = {
        // State
        _initialized: false,
        isConnected: false,
        account: null,
        provider: null,
        wcProvider: null,
        connectionType: null, // 'metamask' or 'walletconnect'
        isProcessingPayment: false, // Prevent double payment
        _txLock: null, // Transaction lock

        // Payment data
        ethAmount: null,
        weiAmount: null,
        usdcAmount: null,
        usdcSmallestUnit: null,

        // Config (will be populated from DOM)
        config: {
            networks: {},
            isTestMode: false,
            merchantAddress: '',
            orderTotal: 0,
            currency: 'EUR',
            convertUrl: '',
            validationUrl: '',
            walletConnectId: '',
            cartId: ''
        },

        // Contract function selectors
        SELECTORS: {
            pay: '0x31cbf5e3', // pay(uint256,address)
            payWithToken: '0xd6bcaa76', // payWithToken(uint256,address,address,uint256)
            approve: '0x095ea7b3', // approve(address,uint256)
            balanceOf: '0x70a08231', // balanceOf(address)
            allowance: '0xdd62ed3e' // allowance(address,address)
        },

        /**
         * Initialize the payment form
         */
        init: function() {
            // Prevent double initialization
            if (this._initialized) {
                console.log('LayerPay.init() already called, skipping');
                return;
            }
            this._initialized = true;

            const container = document.getElementById('layerpay-payment-container');
            if (!container) return;

            // Clean up any stale WalletConnect sessions on fresh page load
            this.cleanupStaleConnections();

            // Parse config from data attributes
            this.config.networks = JSON.parse(container.dataset.networks || '{}');
            this.config.isTestMode = container.dataset.testMode === '1';
            this.config.merchantAddress = container.dataset.merchantAddress;
            this.config.orderTotal = parseFloat(container.dataset.orderTotal);
            this.config.currency = container.dataset.currency;
            this.config.convertUrl = container.dataset.convertUrl;
            this.config.validationUrl = container.dataset.validationUrl;
            this.config.walletConnectId = container.dataset.walletconnectId;
            this.config.cartId = container.dataset.cartId;

            this.bindEvents();
            this.fetchConversion();
            this.checkExistingConnection();
        },

        /**
         * Check if Web3Modal has an existing connection
         */
        checkExistingConnection: function() {
            const self = this;

            // Wait for Web3Modal to be ready
            const checkModal = () => {
                if (window.web3Modal) {
                    const address = window.web3Modal.getAddress();
                    const provider = window.web3Modal.getWalletProvider();

                    if (address) {
                        self.account = address;
                        self.provider = provider;
                        self.wcProvider = provider;
                        self.isConnected = true;
                        self.connectionType = 'walletconnect';
                        self.updateWalletUI();
                    }

                    // Subscribe to future changes (only once globally)
                    if (!window._web3ModalSubscribed) {
                        window._web3ModalSubscribed = true;
                        window.web3Modal.subscribeProvider(({ address, isConnected, provider }) => {
                            if (isConnected && address && provider) {
                                self.account = address;
                                self.provider = provider;
                                self.wcProvider = provider;
                                self.isConnected = true;
                                self.connectionType = 'walletconnect';
                                self.updateWalletUI();
                            } else if (!isConnected) {
                                self.isConnected = false;
                                self.account = null;
                                self.provider = null;
                                self.updateWalletUI();
                            }
                        });
                    }
                } else if (window.web3ModalReady === undefined) {
                    // Web3Modal not loaded yet, wait
                    setTimeout(checkModal, 500);
                }
            };

            // Start checking
            if (window.web3ModalReady) {
                checkModal();
            } else {
                window.addEventListener('web3ModalReady', checkModal, { once: true });
                // Also check after a delay in case event was missed
                setTimeout(checkModal, 1000);
            }
        },

        /**
         * Bind event listeners
         */
        bindEvents: function() {
            // Network selection
            document.querySelectorAll('input[name="layerpay_network_select"]').forEach(input => {
                input.addEventListener('change', (e) => this.onNetworkChange(e.target.value));
            });

            // Payment type selection
            document.querySelectorAll('input[name="layerpay_payment_type"]').forEach(input => {
                input.addEventListener('change', (e) => this.onPaymentTypeChange(e.target.value));
            });

            // Connect button - opens Web3Modal directly
            const connectBtn = document.getElementById('layerpay-connect-btn');
            if (connectBtn && !connectBtn._layerpayBound) {
                connectBtn._layerpayBound = true;
                connectBtn.addEventListener('click', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('Connect button clicked');
                    this.openWeb3Modal();
                }, { capture: true });
            }

            // Pay button - prevent double click and form bubbling
            const payBtn = document.getElementById('layerpay-pay-btn');
            if (payBtn && !payBtn._layerpayBound) {
                payBtn._layerpayBound = true;
                payBtn.addEventListener('click', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('Pay button clicked, isProcessing:', this.isProcessingPayment);
                    this.initiatePayment();
                }, { capture: true });
            }

            // Update UI on network/type change
            this.updateNetworkUI();
            this.updatePaymentTypeUI();
        },

        /**
         * Hide wallet selection modal (legacy, kept for compatibility)
         */
        hideWalletModal: function() {
            // No-op - Web3Modal handles its own modal
        },

        /**
         * Open Web3Modal directly
         */
        openWeb3Modal: async function() {
            if (this.isConnected) {
                // Already connected - open Account view to disconnect
                if (window.web3Modal) {
                    window.web3Modal.open({ view: 'Account' });
                }
                return;
            }

            await this.connectWalletConnect();
        },

        /**
         * Connect using MetaMask
         */
        connectMetaMask: async function() {
            // Find MetaMask provider (handle multiple injected wallets)
            let provider = null;

            if (window.ethereum) {
                // Check if there are multiple providers
                if (window.ethereum.providers && window.ethereum.providers.length > 0) {
                    // Find MetaMask specifically
                    provider = window.ethereum.providers.find(p => p.isMetaMask && !p.isBraveWallet);
                    if (!provider) {
                        provider = window.ethereum.providers[0];
                    }
                } else if (window.ethereum.isMetaMask) {
                    provider = window.ethereum;
                } else {
                    provider = window.ethereum;
                }
            }

            if (!provider) {
                throw new Error('MetaMask not installed. Please install MetaMask extension.');
            }

            try {
                const accounts = await provider.request({
                    method: 'eth_requestAccounts'
                });

                if (!accounts || accounts.length === 0) {
                    throw new Error('No accounts found');
                }

                this.account = accounts[0];
                this.provider = provider;

                // Listen for account changes
                provider.on('accountsChanged', (accounts) => {
                    if (accounts.length === 0) {
                        this.disconnect();
                    } else {
                        this.account = accounts[0];
                        this.updateWalletUI();
                    }
                });

                // Listen for chain changes
                provider.on('chainChanged', () => {
                    this.ensureCorrectNetwork();
                });
            } catch (err) {
                if (err.code === 4001) {
                    throw new Error('Connection rejected by user');
                }
                throw err;
            }
        },

        /**
         * Connect using Web3Modal (pre-loaded in head)
         */
        connectWalletConnect: async function() {
            // Prevent double execution
            if (this._wcConnecting) {
                return;
            }
            this._wcConnecting = true;

            try {
                // Wait for Web3Modal to be ready
                if (!window.web3Modal) {
                    this.updateStatus('Loading Web3Modal...');
                    await new Promise((resolve, reject) => {
                        if (window.web3ModalReady) {
                            resolve();
                            return;
                        }
                        const timeout = setTimeout(() => reject(new Error('Web3Modal load timeout')), 10000);
                        window.addEventListener('web3ModalReady', () => {
                            clearTimeout(timeout);
                            resolve();
                        }, { once: true });
                    });
                }

                const modal = window.web3Modal;

                // Open the modal - subscription in checkExistingConnection handles connection events
                modal.open();

                // Just wait a moment for modal to open, then return
                // The subscription will handle the actual connection
                await new Promise(resolve => setTimeout(resolve, 500));

            } catch (err) {
                // Only throw for real errors, not user actions
                if (err.message && err.message.includes('timeout')) {
                    throw err;
                }
                // Silently ignore modal close - subscription will handle connection
            } finally {
                this._wcConnecting = false;
            }
        },

        /**
         * Clean up stale WalletConnect/Web3Modal sessions
         * Called on page load to prevent connection issues
         */
        cleanupStaleConnections: function() {
            try {
                // Check if we have a session but no valid provider
                const hasSession = Object.keys(localStorage).some(key =>
                    key.startsWith('wc@') || key.startsWith('@w3m') || key.startsWith('W3M')
                );

                if (hasSession) {
                    console.log('LayerPay: Cleaning up stale wallet sessions...');
                    // Clear all WalletConnect/Web3Modal storage
                    Object.keys(localStorage).forEach(key => {
                        if (key.startsWith('wc@') || key.startsWith('@w3m') || key.startsWith('W3M')) {
                            localStorage.removeItem(key);
                        }
                    });
                    // Reset global subscription flag
                    window._web3ModalSubscribed = false;
                }
            } catch (e) {
                console.log('LayerPay: Error cleaning up sessions:', e);
            }
        },

        /**
         * Disconnect wallet
         */
        disconnect: async function() {
            // Try to disconnect from Web3Modal/provider
            if (this.wcProvider) {
                try {
                    if (typeof this.wcProvider.disconnect === 'function') {
                        await this.wcProvider.disconnect();
                    }
                } catch (e) {
                    console.log('WC provider disconnect error:', e);
                }
                this.wcProvider = null;
            }

            // Clear Web3Modal session storage
            try {
                // Clear WalletConnect sessions
                Object.keys(localStorage).forEach(key => {
                    if (key.startsWith('wc@') || key.startsWith('@w3m') || key.startsWith('W3M')) {
                        localStorage.removeItem(key);
                    }
                });
            } catch (e) {
                console.log('Error clearing storage:', e);
            }

            // Reset local state
            this.provider = null;
            this.account = null;
            this.isConnected = false;
            this.connectionType = null;
            this.updateWalletUI();

            // Reload page to fully reset Web3Modal state
            window.location.reload();
        },

        /**
         * Ensure wallet is on the correct network
         */
        ensureCorrectNetwork: async function() {
            if (!this.provider) return;

            const selectedNetwork = this.getSelectedNetwork();
            if (!selectedNetwork) return;

            const networkConfig = this.config.networks[selectedNetwork];
            if (!networkConfig) return;

            try {
                const currentChainId = await this.provider.request({ method: 'eth_chainId' });

                if (currentChainId.toLowerCase() !== networkConfig.chain_id.toLowerCase()) {
                    await this.switchNetwork(networkConfig);
                }
            } catch (error) {
                console.error('Network switch error:', error);
            }
        },

        /**
         * Switch to the specified network
         */
        switchNetwork: async function(networkConfig) {
            try {
                await this.provider.request({
                    method: 'wallet_switchEthereumChain',
                    params: [{ chainId: networkConfig.chain_id }]
                });
            } catch (switchError) {
                // Network not added, try to add it
                if (switchError.code === 4902) {
                    await this.addNetwork(networkConfig);
                } else {
                    throw switchError;
                }
            }
        },

        /**
         * Add network to wallet
         */
        addNetwork: async function(networkConfig) {
            const params = {
                chainId: networkConfig.chain_id,
                chainName: networkConfig.name,
                rpcUrls: [networkConfig.rpc_url],
                nativeCurrency: {
                    name: 'Ethereum',
                    symbol: networkConfig.currency,
                    decimals: 18
                },
                blockExplorerUrls: [networkConfig.explorer]
            };

            await this.provider.request({
                method: 'wallet_addEthereumChain',
                params: [params]
            });
        },

        /**
         * Get selected network key
         */
        getSelectedNetwork: function() {
            const selected = document.querySelector('input[name="layerpay_network_select"]:checked');
            return selected ? selected.value : null;
        },

        /**
         * Get selected payment type
         */
        getSelectedPaymentType: function() {
            const selected = document.querySelector('input[name="layerpay_payment_type"]:checked');
            return selected ? selected.value : 'eth';
        },

        /**
         * Handle network change
         */
        onNetworkChange: function(networkKey) {
            this.updateNetworkUI();
            if (this.isConnected) {
                this.ensureCorrectNetwork();
            }
            this.fetchConversion();
        },

        /**
         * Handle payment type change
         */
        onPaymentTypeChange: function(paymentType) {
            this.updatePaymentTypeUI();
            this.fetchConversion();
        },

        /**
         * Update network selection UI
         */
        updateNetworkUI: function() {
            document.querySelectorAll('.layerpay-network-option').forEach(option => {
                const input = option.querySelector('input');
                option.classList.toggle('selected', input.checked);
            });
        },

        /**
         * Update payment type UI
         */
        updatePaymentTypeUI: function() {
            document.querySelectorAll('.layerpay-type-option').forEach(option => {
                const input = option.querySelector('input');
                option.classList.toggle('selected', input.checked);
            });
        },

        /**
         * Update wallet connection UI
         */
        updateWalletUI: function() {
            const statusEl = document.getElementById('layerpay-wallet-status');
            const connectBtn = document.getElementById('layerpay-connect-btn');
            const payBtn = document.getElementById('layerpay-pay-btn');

            if (this.isConnected && this.account) {
                const shortAddress = this.account.slice(0, 6) + '...' + this.account.slice(-4);

                statusEl.className = 'layerpay-wallet-connected';
                statusEl.innerHTML = `
                    <span class="layerpay-wallet-icon">✓</span>
                    <span class="layerpay-wallet-text">Connected: ${shortAddress}</span>
                    <button type="button" class="layerpay-disconnect-btn" onclick="LayerPay.disconnect()">Disconnect</button>
                `;

                connectBtn.style.display = 'none';
                payBtn.style.display = 'block';

                // Update hidden field
                document.getElementById('layerpay-payer-address').value = this.account;
            } else {
                statusEl.className = 'layerpay-wallet-disconnected';
                statusEl.innerHTML = `
                    <span class="layerpay-wallet-icon">🔗</span>
                    <span class="layerpay-wallet-text">No wallet connected</span>
                `;

                connectBtn.style.display = 'block';
                payBtn.style.display = 'none';
            }
        },

        /**
         * Fetch price conversion
         */
        fetchConversion: async function() {
            const paymentType = this.getSelectedPaymentType();
            const url = this.config.convertUrl +
                '?amount=' + this.config.orderTotal +
                '&currency=' + this.config.currency.toLowerCase() +
                '&type=' + paymentType;

            try {
                const response = await fetch(url);
                const data = await response.json();

                if (data.success) {
                    this.updatePriceDisplay(data.data, paymentType);
                } else {
                    console.error('Conversion error:', data.error);
                }
            } catch (error) {
                console.error('Fetch conversion error:', error);
            }
        },

        /**
         * Update price display
         */
        updatePriceDisplay: function(data, paymentType) {
            const priceSection = document.querySelector('.layerpay-price-section');
            const cryptoAmountEl = document.getElementById('layerpay-crypto-amount');
            const rateEl = document.getElementById('layerpay-rate');

            if (paymentType === 'eth') {
                this.ethAmount = data.eth_amount;
                this.weiAmount = data.wei_amount;
                cryptoAmountEl.textContent = data.eth_amount + ' ETH';
                rateEl.textContent = '1 ETH = ' + data.eth_price + ' ' + data.currency;
            } else {
                this.usdcAmount = data.usdc_amount;
                this.usdcSmallestUnit = data.usdc_smallest_unit;
                cryptoAmountEl.textContent = data.usdc_amount + ' USDC';
                rateEl.textContent = '1 USDC = 1 USD';
            }

            priceSection.style.display = 'block';
        },

        /**
         * Initiate payment
         */
        initiatePayment: async function() {
            console.log('initiatePayment called, isProcessing:', this.isProcessingPayment, '_txLock:', this._txLock);

            // Prevent double execution
            if (this.isProcessingPayment) {
                console.log('Payment already in progress, ignoring duplicate call');
                return;
            }

            if (this._txLock) {
                console.log('Transaction lock active, ignoring:', this._txLock);
                return;
            }

            // Check connection - also check Web3Modal state as fallback
            if (!this.isConnected && !this.account) {
                // Try to get from Web3Modal
                if (window.web3Modal) {
                    const address = window.web3Modal.getAddress();
                    const provider = window.web3Modal.getWalletProvider();
                    if (address && provider) {
                        this.account = address;
                        this.provider = provider;
                        this.isConnected = true;
                    }
                }
            }

            if (!this.isConnected || !this.account) {
                this.showError('Please connect your wallet first');
                return;
            }

            const paymentType = this.getSelectedPaymentType();
            const networkKey = this.getSelectedNetwork();
            const networkConfig = this.config.networks[networkKey];

            if (!networkConfig) {
                this.showError('Invalid network selected');
                return;
            }

            // Lock payment processing
            this.isProcessingPayment = true;
            this.disablePayButton();

            this.updateStatus('Preparing transaction...');

            try {
                // Ensure correct network
                await this.ensureCorrectNetwork();

                let txHash;
                if (paymentType === 'eth') {
                    txHash = await this.sendEthPayment(networkConfig);
                } else {
                    txHash = await this.sendUsdcPayment(networkConfig);
                }

                this.updateStatus('Waiting for confirmation...');
                await this.waitForTransaction(txHash);

                this.updateStatus('Completing order...');
                await this.completeOrder(txHash, networkKey, paymentType);

            } catch (error) {
                console.error('Payment error:', error);
                this.showError('Payment failed: ' + (error.message || 'Unknown error'));
                this.hideStatus();
                // Re-enable payment on error
                this.isProcessingPayment = false;
                this._txLock = null;
                this.enablePayButton();
            }
        },

        /**
         * Disable pay button during processing
         */
        disablePayButton: function() {
            const payBtn = document.getElementById('layerpay-pay-btn');
            if (payBtn) {
                payBtn.disabled = true;
                payBtn.classList.add('disabled');
            }
        },

        /**
         * Re-enable pay button after error
         */
        enablePayButton: function() {
            const payBtn = document.getElementById('layerpay-pay-btn');
            if (payBtn) {
                payBtn.disabled = false;
                payBtn.classList.remove('disabled');
            }
        },

        /**
         * Send ETH payment
         */
        sendEthPayment: async function(networkConfig) {
            // Transaction-level lock
            const txId = `eth-${this.config.cartId}-${Date.now()}`;
            if (this._txLock) {
                console.log('Transaction already in progress:', this._txLock);
                throw new Error('Transaction already in progress');
            }
            this._txLock = txId;
            console.log('Starting ETH transaction:', txId);

            // Verify we're on the correct network
            const currentChainId = await this.provider.request({ method: 'eth_chainId' });
            console.log('Current chain ID:', currentChainId, 'Expected:', networkConfig.chain_id);
            if (currentChainId.toLowerCase() !== networkConfig.chain_id.toLowerCase()) {
                throw new Error(`Wrong network! Connected to ${currentChainId}, expected ${networkConfig.chain_id}. Please switch network in your wallet.`);
            }

            const orderId = parseInt(this.config.cartId);
            const merchantAddress = this.config.merchantAddress.slice(2).toLowerCase();

            // Encode function call: pay(uint256 orderId, address merchantAddress)
            const data = this.SELECTORS.pay +
                this.padHex(orderId.toString(16), 64) +
                this.padHex(merchantAddress, 64);

            const tx = {
                from: this.account,
                to: networkConfig.contract,
                value: '0x' + BigInt(this.weiAmount).toString(16),
                data: data
            };

            const txHash = await this.provider.request({
                method: 'eth_sendTransaction',
                params: [tx]
            });

            return txHash;
        },

        /**
         * Send USDC payment
         */
        sendUsdcPayment: async function(networkConfig) {
            // Transaction-level lock
            const txId = `usdc-${this.config.cartId}-${Date.now()}`;
            if (this._txLock) {
                console.log('Transaction already in progress:', this._txLock);
                throw new Error('Transaction already in progress');
            }
            this._txLock = txId;
            console.log('Starting USDC transaction:', txId);

            const usdcAddress = networkConfig.usdc_address;
            const contractAddress = networkConfig.contract;
            const amount = BigInt(this.usdcSmallestUnit);

            console.log('USDC payment details:', {
                usdcAddress,
                contractAddress,
                amount: amount.toString(),
                account: this.account
            });

            // CRITICAL: Verify we're on the correct network BEFORE any transactions
            const currentChainId = await this.provider.request({ method: 'eth_chainId' });
            console.log('Current chain ID:', currentChainId, 'Expected:', networkConfig.chain_id);
            if (currentChainId.toLowerCase() !== networkConfig.chain_id.toLowerCase()) {
                throw new Error(`Wrong network! Connected to ${currentChainId}, expected ${networkConfig.chain_id}. Please switch network in your wallet.`);
            }
            console.log('Network verified OK');

            // Check allowance
            console.log('Checking USDC allowance...');
            const allowance = await this.getUsdcAllowance(usdcAddress, contractAddress);
            console.log('Current allowance:', allowance, 'Need:', amount.toString());

            if (BigInt(allowance) < amount) {
                this.updateStatus('Step 1/2: Approving USDC... (confirm in wallet)');
                console.log('Allowance insufficient, requesting approval...');

                // Approve with 10% buffer
                const approvalAmount = amount * BigInt(110) / BigInt(100);
                await this.approveUsdc(usdcAddress, contractAddress, approvalAmount);

                console.log('Approval confirmed, waiting for propagation...');
                // Wait longer for approval to propagate on testnet
                await this.sleep(5000);
            }

            this.updateStatus('Step 2/2: Sending USDC payment... (confirm in wallet)');
            console.log('Sending USDC payment transaction...');

            // Encode function call: payWithToken(uint256 orderId, address merchant, address token, uint256 amount)
            const orderId = parseInt(this.config.cartId);
            const merchantAddress = this.config.merchantAddress.slice(2).toLowerCase();

            const data = this.SELECTORS.payWithToken +
                this.padHex(orderId.toString(16), 64) +
                this.padHex(merchantAddress, 64) +
                this.padHex(usdcAddress.slice(2).toLowerCase(), 64) +
                this.padHex(amount.toString(16), 64);

            const tx = {
                from: this.account,
                to: contractAddress,
                data: data
            };

            // Estimate gas first to catch revert errors early
            console.log('Estimating gas for payWithToken...');
            try {
                const gasEstimate = await this.provider.request({
                    method: 'eth_estimateGas',
                    params: [tx]
                });
                console.log('Gas estimate:', gasEstimate);
                // Add 20% buffer
                tx.gas = '0x' + (BigInt(gasEstimate) * BigInt(120) / BigInt(100)).toString(16);
                console.log('Gas with buffer:', tx.gas);
            } catch (gasErr) {
                console.error('Gas estimation failed:', gasErr);
                // Try static call to get better error
                try {
                    await this.provider.request({
                        method: 'eth_call',
                        params: [tx, 'latest']
                    });
                } catch (callErr) {
                    console.error('Static call failed:', callErr);
                    throw new Error('Transaction would fail: ' + (callErr.message || 'Unknown error'));
                }
                throw gasErr;
            }

            const txHash = await this.provider.request({
                method: 'eth_sendTransaction',
                params: [tx]
            });

            return txHash;
        },

        /**
         * Get USDC allowance
         */
        getUsdcAllowance: async function(usdcAddress, spender) {
            const data = this.SELECTORS.allowance +
                this.padHex(this.account.slice(2).toLowerCase(), 64) +
                this.padHex(spender.slice(2).toLowerCase(), 64);

            const result = await this.provider.request({
                method: 'eth_call',
                params: [{
                    to: usdcAddress,
                    data: data
                }, 'latest']
            });

            return result;
        },

        /**
         * Approve USDC spending
         */
        approveUsdc: async function(usdcAddress, spender, amount) {
            console.log('Approving USDC:', { usdcAddress, spender, amount: amount.toString() });

            const data = this.SELECTORS.approve +
                this.padHex(spender.slice(2).toLowerCase(), 64) +
                this.padHex(amount.toString(16), 64);

            console.log('Approve calldata:', data);

            const tx = {
                from: this.account,
                to: usdcAddress,
                data: data
            };

            console.log('Sending approve tx:', tx);

            // Estimate gas first
            try {
                const gasEstimate = await this.provider.request({
                    method: 'eth_estimateGas',
                    params: [tx]
                });
                console.log('Approve gas estimate:', gasEstimate);
                tx.gas = '0x' + (BigInt(gasEstimate) * BigInt(120) / BigInt(100)).toString(16);
            } catch (gasErr) {
                console.error('Approve gas estimation failed:', gasErr);
                // Continue anyway, let wallet estimate
            }

            try {
                const txHash = await this.provider.request({
                    method: 'eth_sendTransaction',
                    params: [tx]
                });

                console.log('Approve tx hash:', txHash);
                await this.waitForTransaction(txHash);
                console.log('Approve confirmed!');
            } catch (err) {
                console.error('Approve error:', err);
                // Try to extract more details
                if (err.data) console.error('Error data:', err.data);
                if (err.code) console.error('Error code:', err.code);
                throw new Error('USDC approval failed: ' + (err.message || 'Unknown error'));
            }
        },

        /**
         * Wait for transaction to be mined
         */
        waitForTransaction: async function(txHash, maxAttempts = 60) {
            console.log('Waiting for transaction:', txHash);

            for (let i = 0; i < maxAttempts; i++) {
                try {
                    const receipt = await this.provider.request({
                        method: 'eth_getTransactionReceipt',
                        params: [txHash]
                    });

                    console.log('Receipt attempt', i + 1, ':', receipt);

                    if (receipt) {
                        // Handle different status formats from different providers
                        // Can be: '0x1', '0x0', 1, 0, true, false, '1', '0'
                        const status = receipt.status;
                        const isSuccess = status === '0x1' || status === 1 || status === '1' || status === true;
                        const isFailed = status === '0x0' || status === 0 || status === '0' || status === false;

                        console.log('Transaction status:', status, '| isSuccess:', isSuccess, '| isFailed:', isFailed);

                        if (isSuccess) {
                            console.log('Transaction confirmed successfully!');
                            return receipt;
                        } else if (isFailed) {
                            throw new Error('Transaction reverted on blockchain');
                        } else {
                            // Unknown status format, log and assume success if we have a receipt
                            console.warn('Unknown status format:', status, '- assuming success');
                            return receipt;
                        }
                    }
                } catch (err) {
                    // If it's our own error, rethrow it
                    if (err.message && (err.message.includes('reverted') || err.message.includes('failed'))) {
                        throw err;
                    }
                    // Otherwise it's a network error, continue polling
                    console.log('Receipt fetch error, retrying...', err.message);
                }

                await this.sleep(2000);
            }

            throw new Error('Transaction confirmation timeout');
        },

        /**
         * Complete the order
         */
        completeOrder: async function(txHash, networkKey, paymentType) {
            const cryptoAmount = paymentType === 'eth' ? this.ethAmount : this.usdcAmount;

            // Update hidden fields
            document.getElementById('layerpay-tx-hash').value = txHash;
            document.getElementById('layerpay-network').value = networkKey;
            document.getElementById('layerpay-crypto-amount-input').value = cryptoAmount;
            document.getElementById('layerpay-payment-type-input').value = paymentType;

            // Send validation request
            const formData = new FormData();
            formData.append('layerpay_tx_hash', txHash);
            formData.append('layerpay_network', networkKey);
            formData.append('layerpay_crypto_amount', cryptoAmount);
            formData.append('layerpay_payment_type', paymentType);
            formData.append('layerpay_payer_address', this.account);
            formData.append('ajax', '1');

            const response = await fetch(this.config.validationUrl, {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success && result.data.redirect_url) {
                window.location.href = result.data.redirect_url;
            } else {
                throw new Error(result.error || 'Order validation failed');
            }
        },

        /**
         * Utility: Pad hex string
         */
        padHex: function(hex, length) {
            return hex.padStart(length, '0');
        },

        /**
         * Utility: Sleep
         */
        sleep: function(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        },

        /**
         * Show status message
         */
        updateStatus: function(message) {
            const statusEl = document.getElementById('layerpay-tx-status');
            const textEl = document.getElementById('layerpay-status-text');

            if (statusEl && textEl) {
                textEl.textContent = message;
                statusEl.style.display = 'block';
            }
        },

        /**
         * Hide status message
         */
        hideStatus: function() {
            const statusEl = document.getElementById('layerpay-tx-status');
            if (statusEl) statusEl.style.display = 'none';
        },

        /**
         * Show error message
         */
        showError: function(message) {
            alert(message); // Simple implementation, can be improved
        }
    };

    // Make LayerPay globally accessible
    window.LayerPay = LayerPay;

    // Initialize on DOM ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => LayerPay.init());
    } else {
        LayerPay.init();
    }
})();
