<?php
/**
 * LayerPay Validation Controller
 * Handles payment validation and order creation
 */

class LayerPayValidationModuleFrontController extends ModuleFrontController
{
    public $ssl = true;

    public function postProcess()
    {
        // Check if module is active
        if (!$this->module->active) {
            $this->ajaxError('Module is not active');
        }

        // Get cart
        $cart = $this->context->cart;
        if (!Validate::isLoadedObject($cart) || $cart->nbProducts() < 1) {
            $this->ajaxError('Cart is empty or invalid');
        }

        // Get POST data
        $txHash = Tools::getValue('layerpay_tx_hash');
        $network = Tools::getValue('layerpay_network');
        $cryptoAmount = Tools::getValue('layerpay_crypto_amount');
        $paymentType = Tools::getValue('layerpay_payment_type');
        $payerAddress = Tools::getValue('layerpay_payer_address');

        // Validate transaction hash
        if (!preg_match('/^0x[a-fA-F0-9]{64}$/', $txHash)) {
            $this->ajaxError('Invalid transaction hash');
        }

        // Validate network
        $networkData = $this->module->getNetwork($network);
        if (!$networkData) {
            $this->ajaxError('Invalid network');
        }

        // Validate payment type
        if (!in_array($paymentType, ['eth', 'usdc'])) {
            $this->ajaxError('Invalid payment type');
        }

        // Get order total
        $currency = new Currency($cart->id_currency);
        $total = $cart->getOrderTotal(true, Cart::BOTH);

        // Verify transaction on blockchain
        $verification = $this->module->verifyTransactionOnchain(
            $txHash,
            $network,
            $cryptoAmount,
            $paymentType
        );

        if (!$verification['success']) {
            $this->module->log('Verification failed: ' . $verification['error']);
            $this->ajaxError('Transaction verification failed: ' . $verification['error']);
        }

        // Create the order
        $customer = new Customer($cart->id_customer);
        if (!Validate::isLoadedObject($customer)) {
            $this->ajaxError('Customer not found');
        }

        $paymentMethod = sprintf(
            'LayerPay (%s on %s)',
            strtoupper($paymentType),
            $networkData['name']
        );

        try {
            // Validate order
            $this->module->validateOrder(
                $cart->id,
                Configuration::get('PS_OS_PAYMENT'),
                $total,
                $paymentMethod,
                sprintf(
                    'Crypto payment: %s %s on %s. TX: %s',
                    $cryptoAmount,
                    strtoupper($paymentType),
                    $networkData['name'],
                    $txHash
                ),
                [
                    'transaction_id' => $txHash
                ],
                (int)$currency->id,
                false,
                $customer->secure_key
            );

            $orderId = $this->module->currentOrder;

            // Save payment data
            $this->module->saveOrderPaymentData($orderId, [
                'tx_hash' => $txHash,
                'network' => $network,
                'crypto_amount' => $cryptoAmount,
                'payment_type' => $paymentType,
                'payer_address' => $payerAddress,
                'verified' => 1,
                'verified_payer' => $verification['data']['payer'] ?? '',
                'verified_amount' => $verification['data']['amount'] ?? 0,
                'block_number' => $verification['data']['block_number'] ?? 0
            ]);

            // Redirect to confirmation page
            $order = new Order($orderId);

            if (Tools::getValue('ajax')) {
                $this->ajaxSuccess([
                    'order_id' => $orderId,
                    'order_reference' => $order->reference,
                    'redirect_url' => $this->context->link->getPageLink(
                        'order-confirmation',
                        true,
                        null,
                        [
                            'id_cart' => $cart->id,
                            'id_module' => $this->module->id,
                            'id_order' => $orderId,
                            'key' => $customer->secure_key
                        ]
                    )
                ]);
            } else {
                Tools::redirect($this->context->link->getPageLink(
                    'order-confirmation',
                    true,
                    null,
                    [
                        'id_cart' => $cart->id,
                        'id_module' => $this->module->id,
                        'id_order' => $orderId,
                        'key' => $customer->secure_key
                    ]
                ));
            }
        } catch (Exception $e) {
            $this->module->log('Order creation error: ' . $e->getMessage());
            $this->ajaxError('Failed to create order: ' . $e->getMessage());
        }
    }

    protected function ajaxSuccess($data)
    {
        header('Content-Type: application/json');
        die(json_encode([
            'success' => true,
            'data' => $data
        ]));
    }

    protected function ajaxError($message)
    {
        if (Tools::getValue('ajax')) {
            header('Content-Type: application/json');
            die(json_encode([
                'success' => false,
                'error' => $message
            ]));
        } else {
            $this->errors[] = $message;
            $this->redirectWithNotifications($this->context->link->getPageLink('order', true, null, ['step' => 3]));
        }
    }
}
